LoginManager.cs


using UnityEngine;
using UnityEngine.UI;
using UnityEngine.SceneManagement;
using UnityEngine.Networking;
using System.Collections;
using System.Collections.Generic;
using TMPro;

public class LoginManager : MonoBehaviour
{
    [Header("UI References")]
    public InputField idInputField;            // ユーザー入力欄
    public Button loginButton;                // ログインボタン
    public InputField errorInputField;        // エラーメッセージ表示用InputField
    public GameObject suggestionPanel;        // 履歴候補を表示するPanel
    public GameObject suggestionButtonPrefab; // 履歴候補用ボタンプレハブ
    public Transform suggestionContent;       // ScrollViewのContent部分

    private List<string> inputHistory = new List<string>(); // 入力履歴リスト
    private string historyKey = "InputHistoryKey"; // PlayerPrefsのキー

    private void Start()
    {
        // 履歴の読み込みと初期化
        LoadInputHistory();

        // 空文字で候補を表示（初期状態で全履歴を表示）
        UpdateSuggestions("");

        // イベント設定
        idInputField.onValueChanged.AddListener(OnInputChanged);
        loginButton.onClick.AddListener(AttemptLogin);

        // 初期状態では候補リストを表示
        suggestionPanel.SetActive(true);
    }

    private void Update()
    {
        // Enterキーが押されたらAttemptLoginを呼び出す
        if (Input.GetKeyDown(KeyCode.Return))
        {
            AttemptLogin();
        }
    }

    private void AttemptLogin()
    {
        string id = idInputField.text; // ユーザーが入力したIDを取得
        string url = $"https://kodamalab.sakura.ne.jp/Shigeta/Shigeta/{id}/{id}_HelloFitbit.png";

        StartCoroutine(CheckIfImageExists(url, id));
    }

    private IEnumerator CheckIfImageExists(string url, string id)
    {
        using (UnityWebRequest uwr = UnityWebRequest.Head(url))
        {
            yield return uwr.SendWebRequest();

            if (uwr.result == UnityWebRequest.Result.ConnectionError || uwr.result == UnityWebRequest.Result.ProtocolError)
            {
                errorInputField.text = "Error: IDが見つかりません。";
            }
            else
            {
                string successMessage = "Hello FitbitAgent!";
                errorInputField.text = successMessage;
                Debug.Log(successMessage);

                // 履歴保存
                SaveInputHistory(id);

                // IDを保存し次のシーンに遷移
                PlayerPrefs.SetString("userID", id);
                SceneManager.LoadScene("AdviceGPT4o");
            }
        }
    }

    // 入力履歴の保存
    private void SaveInputHistory(string newInput)
    {
        if (!string.IsNullOrEmpty(newInput) && !inputHistory.Contains(newInput))
        {
            inputHistory.Add(newInput);
            PlayerPrefs.SetString(historyKey, string.Join(",", inputHistory));
            PlayerPrefs.Save();
        }
    }

    // 履歴の読み込み
    private void LoadInputHistory()
    {
        if (PlayerPrefs.HasKey(historyKey))
        {
            string historyString = PlayerPrefs.GetString(historyKey);
            inputHistory = new List<string>(historyString.Split(','));
        }
    }

    // 入力が変更されたときに履歴候補を表示
    private void OnInputChanged(string input)
    {
        UpdateSuggestions(input);
    }

    private void UpdateSuggestions(string input)
    {
        // 既存の候補ボタンを削除
        foreach (Transform child in suggestionContent)
        {
            Destroy(child.gameObject);
        }

        // 入力が空の場合、全履歴を表示
        List<string> filteredSuggestions = string.IsNullOrEmpty(input)
            ? inputHistory // 空ならすべての履歴
            : inputHistory.FindAll(item => item.StartsWith(input));

        // 候補がない場合はPanelを非表示に
        if (filteredSuggestions.Count == 0)
        {
            suggestionPanel.SetActive(false);
            return;
        }

        // 候補がある場合は表示
        suggestionPanel.SetActive(true);

        // 履歴候補ボタンを生成
        foreach (string suggestion in filteredSuggestions)
        {
            GameObject buttonObj = Instantiate(suggestionButtonPrefab, suggestionContent);
            buttonObj.GetComponentInChildren<TextMeshProUGUI>().text = suggestion;

            // 候補が選択された場合、InputFieldに反映
            buttonObj.GetComponent<Button>().onClick.AddListener(() =>
            {
                idInputField.text = suggestion;
                suggestionPanel.SetActive(false); // 候補リストを非表示に
            });
        }
    }
}
