// HTTP‑ECG21 Dual‑Core Version
// =============================================================
// 1) ベース：最初に頂いた HTTP‑ECG21 スケッチ
// 2) 追加：『デュアルタスク化』スケッチの構成（ECG Task / HTTP Task）
//    ‑‑ 重複する部分はこのデュアルタスク版を優先
// 3) PHP 側 URL／パラメータ形式は最初のスケッチのまま維持
// -------------------------------------------------------------

#include <Wire.h>                  // I2C 通信
#include <Adafruit_ADS1X15.h>      // ADS1015 / ADS1115
#include <WiFi.h>                  // Wi‑Fi (ESP32)
#include <HTTPClient.h>            // HTTP クライアント
#include <time.h>                  // NTP
#include <freertos/FreeRTOS.h>
#include <freertos/task.h>
#include <freertos/queue.h>
#include <cmath>
#include <string.h>

// =============================================================
// ★ プログラム定数（元コード＋仕様変更）
// -------------------------------------------------------------
#define M                 15
#define N                 45
#define winSize           250
#define HP_CONSTANT       ((float)1 / (float)M)
#define WINDOW_SIZE       16
#define QUEUE_LENGTH      50
#define REFRACT_MS        150      // Refractory period (ms)
#define AMP_THRESH        4.5f     // Amplitude threshold (|HP| > x)

// 送信キューの最大深さ（リトライ用に少し多め）
#define RETRY_DEPTH       50

// 外れ値抑制用
#define MEDIAN_WIN   9
#define OUTLIER_PCT  0.50f  // ±50 % 超は外れ値

// サンプリング周期（ms）─ 1kHz ≒ 1ms。500Hz にしたいなら 2 に変更
#define SAMPLE_INTERVAL_MS 1

// ===== RMSSD 用（直近1分） =====
#define RMSSD_WINDOW_MS  (60000UL) // 1分＝60,000ms
#define RMSSD_MAX_BEATS  180       // 窓内最大拍数（180 bpm を上限想定）

// =============================================================
// Wi‑Fi / サーバー設定（★ PHP 側 URL は最初のまま維持）
// -------------------------------------------------------------
const char* ssid       = "HELL";
const char* password   = "kodamamasahisa";

const char* serverUrl  = "http://kodamalab.sakura.ne.jp/Shigeta/HTTPECG/Record.php";
const char* controlUrl = "http://kodamalab.sakura.ne.jp/Shigeta/HTTPECG/SoundControl.php";
const char* deviceID   = "ECG02";   // デバイス ID

// NTP サーバー（JST）
const char* ntpServer     = "ntp.nict.jp";
const long  gmtOffset     = 9 * 3600;
const int   daylightOffset = 0;

// ピン定義
#define BUTTON_PIN      39
#define BUZZER_PIN      19

// =============================================================
// グローバルオブジェクト／変数
// -------------------------------------------------------------
Adafruit_ADS1015 ads;                 // ADS1015 インスタンス

// ECG バッファ & 微分用
float voltageBuffer[WINDOW_SIZE] = {0};
float weights[WINDOW_SIZE] = {
  -8, -7, -6, -5, -4, -3, -2, -1,
   1,  2,  3,  4,  5,  6,  7,  8
};

// QRS 検出用ステート
unsigned long previousMillis = 0;   // 直近 R 波検出時刻 (ms)

// ====== RMSSD：1分窓リングバッファ ======
// 拍の検出「時刻」と「IBI」を同順で保持
static unsigned long rmssdBeatTimes[RMSSD_MAX_BEATS] = {0};
static unsigned long rmssdIbis[RMSSD_MAX_BEATS]      = {0};
static int rmssdStart = 0;  // 先頭インデックス
static int rmssdCount = 0;  // 要素数（0..RMSSD_MAX_BEATS）
float latestRmssd = 0.0f;   // 直近のRMSSD（1分窓から）

// 外れ値抑制用バッファ（中央値フィルタ用）
static unsigned long ibiMedBuf[MEDIAN_WIN] = {0};
static int ibiMedIdx = 0, ibiMedCnt = 0;

// サウンド ON/OFF フラグ（ボタン or HTTP で変化）
volatile bool soundEnabled = true;

// 送信パケット構造体
struct ECGData {
  time_t        timestamp;  // Epoch seconds
  unsigned long ibi;        // RR (ms)
  float         hr;         // Heart rate (bpm)
  float         rmssd;      // HRV (RMSSD)
};

// 送信用キュー
static QueueHandle_t ecgQueue;

// =============================================================
// 関数プロトタイプ
// -------------------------------------------------------------
static boolean detect(float new_ecg_pt);
static float    smoothdiff(float newVoltage);
static bool     sendData(const ECGData &d);

// ★ RMSSD 1分窓管理
static void     rmssdWindowPush(unsigned long beatTimeMs, unsigned long ibiMs);
static float    computeRmssdFromWindow();

void ecgTask(void* pvParameters);   // Core0：ECG 取得 & 解析
void httpTask(void* pvParameters);  // Core1：HTTP 送信
void soundControlTask(void* pvParameters); // Core1：サウンド遠隔制御

// =============================================================
// setup()
// -------------------------------------------------------------
void setup() {
  // ESP32 動作周波数を 160 MHz に固定
  setCpuFrequencyMhz(160);

  Serial.begin(115200);
  pinMode(BUZZER_PIN, OUTPUT);
  pinMode(BUTTON_PIN, INPUT_PULLUP);

  // I2C & ADC 初期化
  Wire.begin(21, 25);
  ads.begin();

  // Wi‑Fi 接続
  WiFi.begin(ssid, password);
  Serial.print("Connecting to WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print('.');
  }
  Serial.print("\nWiFi connected: ");
  Serial.println(WiFi.localIP());

  // NTP で時刻合わせ
  configTime(gmtOffset, daylightOffset, ntpServer, "pool.ntp.org");
  Serial.print("Waiting for time");
  while (time(nullptr) < 24 * 3600) {
    Serial.print('.');
    delay(500);
  }
  Serial.println("\nTime synchronized");

  // 送信キュー生成
  ecgQueue = xQueueCreate(QUEUE_LENGTH, sizeof(ECGData));

  // -------- デュアルタスク起動 --------
  xTaskCreatePinnedToCore(ecgTask,  "ECGTask",   8 * 1024, nullptr, 1, nullptr, 0); // Core0
  xTaskCreatePinnedToCore(httpTask, "HTTPTask",  8 * 1024, nullptr, 1, nullptr, 1); // Core1
  xTaskCreatePinnedToCore(soundControlTask, "SoundCtl", 4 * 1024, nullptr, 1, nullptr, 1); // Core1
}

// =============================================================
// loop() : 何もしない
// -------------------------------------------------------------
void loop() {
  // すべての処理はタスクへ委譲
  delay(portMAX_DELAY);
}

// =============================================================
// ECG Task  (Core0) ― 信号取得 & 解析
// -------------------------------------------------------------
void ecgTask(void* pvParameters) {
  unsigned long lastSampleMs = millis();

  for (;;) {
    // ── ボタン押下でサウンド ON/OFF ──
    if (digitalRead(BUTTON_PIN) == LOW) {
      soundEnabled = !soundEnabled;
      Serial.printf("Sound %s\n", soundEnabled ? "enabled" : "disabled");
      while (digitalRead(BUTTON_PIN) == LOW) delay(10);
      delay(50);
    }

    // ── サンプリング周期管理 ──
    unsigned long nowMs = millis();
    if (nowMs - lastSampleMs >= SAMPLE_INTERVAL_MS) {
      lastSampleMs += SAMPLE_INTERVAL_MS;

      // ADC 取得（ADS1015：差動 CH0‑CH1）
      int16_t raw     = ads.readADC_Differential_0_1();
      float   new_ecg = -raw * 3.0f;   // ゲイン調整（元コード踏襲）

      // 平滑化微分フィルタ更新（結果は使わないが内部状態更新のため呼ぶ）
      smoothdiff(new_ecg);

      // QRS 検出
      if (detect(new_ecg)) {
        unsigned long currMillis = nowMs;
        unsigned long ibiRaw     = (previousMillis == 0) ? 0 : (currMillis - previousMillis);
        previousMillis           = currMillis;

        // 最初の1拍目は IBI が確定しないためスキップ
        if (ibiRaw == 0) {
          if (soundEnabled) { digitalWrite(BUZZER_PIN, HIGH); delay(20); digitalWrite(BUZZER_PIN, LOW); }
          vTaskDelay(pdMS_TO_TICKS(1));
          continue;
        }

        // ── ブザー鳴動 ──
        if (soundEnabled) {
          digitalWrite(BUZZER_PIN, HIGH);
          delay(20);
          digitalWrite(BUZZER_PIN, LOW);
        }

        // ----- 外れ値抑制（中央値→±50%超は補正） -----
        ibiMedBuf[ibiMedIdx] = ibiRaw;
        ibiMedIdx = (ibiMedIdx + 1) % MEDIAN_WIN;
        if (ibiMedCnt < MEDIAN_WIN) ibiMedCnt++;

        unsigned long medArr[MEDIAN_WIN];
        memcpy(medArr, ibiMedBuf, sizeof(medArr));
        int nMed = ibiMedCnt;
        for (int i = 1; i < nMed; ++i) {
          unsigned long key = medArr[i];
          int j = i - 1;
          while (j >= 0 && medArr[j] > key) { medArr[j + 1] = medArr[j]; --j; }
          medArr[j + 1] = key;
        }
        unsigned long medianIBI = medArr[nMed / 2];

        float dev = fabs((float)ibiRaw - (float)medianIBI) / (medianIBI > 0 ? (float)medianIBI : 1.0f);
        unsigned long ibiCorrected = (dev > OUTLIER_PCT) ? medianIBI : ibiRaw;

        // ----- HR：各拍ごと（平均なし） -----
        float hr = (ibiCorrected > 0) ? (60000.0f / (float)ibiCorrected) : 0.0f;

        // ----- RMSSD：直近 ≤1分窓で再計算 -----
        rmssdWindowPush(currMillis, ibiCorrected);
        if (rmssdCount >= 2) {
          latestRmssd = computeRmssdFromWindow();  // 直近≤1分の連続拍差から算出
          } else {
           latestRmssd = 0.0f;
          }


        // 送信パケット生成 & キュー送信
        ECGData pkt;
        pkt.timestamp = time(nullptr);
        pkt.ibi       = ibiCorrected;
        pkt.hr        = hr;
        pkt.rmssd     = latestRmssd;

        xQueueSend(ecgQueue, &pkt, 0);
      }
    }

    // 自発 yield（必ず）
    vTaskDelay(pdMS_TO_TICKS(1));
  }
}

// =============================================================
// HTTP Task  (Core1) ― データ送信
// -------------------------------------------------------------
void httpTask(void* pvParameters) {
  ECGData retryBuf[RETRY_DEPTH];
  int     retryCnt = 0;

  for (;;) {
    ECGData pkt;
    // キューから取得（ブロック待ち）
    if (xQueueReceive(ecgQueue, &pkt, portMAX_DELAY) == pdTRUE) {
      // 再送前にまず保留分を処理
      for (int i = 0; i < retryCnt; ++i) {
        if (sendData(retryBuf[i])) {
          // 成功 → バッファから外す
          for (int j = i + 1; j < retryCnt; ++j) retryBuf[j - 1] = retryBuf[j];
          --retryCnt; --i;  // インデックス調整
        }
      }

      // 現在のパケット送信
      if (!sendData(pkt)) {
        if (retryCnt < RETRY_DEPTH) retryBuf[retryCnt++] = pkt;
      }
    }
  }
}

// =============================================================
// 遠隔サウンド制御タスク（Core1）
// -------------------------------------------------------------
void soundControlTask(void* pvParameters) {
  HTTPClient http;
  const uint32_t pollInterval = 1000;  // ms

  for (;;) {
    if (WiFi.status() == WL_CONNECTED) {
      String url = String(controlUrl) + "?id=" + deviceID + "&getState=1";
      http.begin(url);
      int code = http.GET();
      if (code == HTTP_CODE_OK) {
        String resp = http.getString();
        resp.trim();
        if (resp == "1")       soundEnabled = true;
        else if (resp == "0")  soundEnabled = false;
      }
      http.end();
    }
    vTaskDelay(pdMS_TO_TICKS(pollInterval));
  }
}

// =============================================================
// sendData() ― データ 1 件送信（成功: true）
// -------------------------------------------------------------
bool sendData(const ECGData &d) {
  if (WiFi.status() != WL_CONNECTED) return false;

  /* ---------- タイムスタンプ整形 ---------- */
  struct tm tm;
  localtime_r(&d.timestamp, &tm);

  char ts_base[20];                   // yyyyMMdd_HHmmss
  strftime(ts_base, sizeof(ts_base), "%Y%m%d_%H%M%S", &tm);

  // 現在ミリ秒（0–999）を追加して “yyyyMMdd_HHmmss_mmm” を作る
  uint16_t ms = millis() % 1000;      // 例: 837
  char ts1[24];
  snprintf(ts1, sizeof(ts1), "%s_%03u", ts_base, ms);

  char ts2[9];
  strftime(ts2, sizeof(ts2), "%H:%M:%S", &tm);

  /* ---------- HTTP 送信 ---------- */
  HTTPClient http;
  String url = String(serverUrl)
             + "?id="     + deviceID
             + "&ts1="    + ts1          // ← ミリ秒付き
             + "&ts2="    + ts2
             + "&rr="     + String(d.ibi)
             + "&hr="     + String(d.hr, 2)
             + "&rmssd="  + String(d.rmssd, 2);

  http.begin(url);
  int code = http.GET();
  http.end();

  return (code == HTTP_CODE_OK);
}


// =============================================================
// QRS 検出アルゴリズム（元コードそのまま）
// -------------------------------------------------------------
boolean detect(float new_ecg_pt) {
  static float ecg_buff[M + 1] = {0};
  static int   wr = 0, rd = 0;
  static float hp_buff[N + 1] = {0};
  static int   hp_w = 0, hp_r = 0;
  static float hp_sum = 0, lp_sum = 0;
  static float threshold = 0, win_max = -1e8;
  static bool  triggered = false;
  static int   trig_time = 0, win_idx = 0, number_iter = 0;

  // ハイパス
  ecg_buff[wr++] = new_ecg_pt; wr %= (M + 1);
  if (number_iter < M) {
    hp_sum += ecg_buff[rd];
    hp_buff[hp_w] = 0.0f;
  } else {
    hp_sum += ecg_buff[rd] - ecg_buff[(rd - M + (M + 1)) % (M + 1)];
    int c = (rd - (M + 1) / 2 + (M + 1)) % (M + 1);
    hp_buff[hp_w] = ecg_buff[c] - HP_CONSTANT * hp_sum;
  }
  rd = (rd + 1) % (M + 1);
  hp_w = (hp_w + 1) % (N + 1);

  // ローパスエネルギー
  lp_sum += hp_buff[hp_r] * hp_buff[hp_r];
  if (number_iter >= N) {
    int old = (hp_r - N + (N + 1)) % (N + 1);
    lp_sum -= hp_buff[old] * hp_buff[old];
  }
  hp_r = (hp_r + 1) % (N + 1);

  // 初期しきい値学習
  if (number_iter < winSize) {
    if (lp_sum > threshold) threshold = lp_sum;
    number_iter++;
  }

  // リフラクトリ期間
  if (triggered) {
    if (++trig_time >= REFRACT_MS) { triggered = false; trig_time = 0; }
  }

  // エネルギー最大値更新
  if (lp_sum > win_max) win_max = lp_sum;

  int lastHpIdx = (hp_w - 1 + (N + 1)) % (N + 1);
  float hp_val = hp_buff[lastHpIdx];

  bool ampOK = (fabs(hp_val) > AMP_THRESH);
  bool engOK = (lp_sum > threshold);
  bool detected = (ampOK && engOK && !triggered);
  if (detected) { triggered = true; trig_time = 0; }

  // しきい値の自動更新
  if (++win_idx >= winSize) {
    float gamma = 0.175f;
    float alpha = 0.01f + ((float)esp_random() / UINT32_MAX) * (0.10f - 0.01f);
    threshold = alpha * gamma * win_max + (1.0f - alpha) * threshold;
    win_idx = 0; win_max = -1e8;
  }

  return detected;
}

// =============================================================
// 平滑化微分関数
// -------------------------------------------------------------
float smoothdiff(float newVoltage) {
  for (int i = WINDOW_SIZE - 1; i > 0; --i) voltageBuffer[i] = voltageBuffer[i - 1];
  voltageBuffer[0] = newVoltage;

  float result = 0.0f;
  for (int i = 0; i < WINDOW_SIZE; ++i) result += voltageBuffer[i] * (-weights[i]);

  return result / (float)WINDOW_SIZE;
}

// =============================================================
// RMSSD 1分窓：管理＆計算
// -------------------------------------------------------------
static void rmssdWindowPush(unsigned long beatTimeMs, unsigned long ibiMs) {
  // 末尾位置（rmssdStart + rmssdCount）に追加
  int end = (rmssdStart + rmssdCount) % RMSSD_MAX_BEATS;
  rmssdBeatTimes[end] = beatTimeMs;
  rmssdIbis[end]      = ibiMs;

  if (rmssdCount < RMSSD_MAX_BEATS) {
    rmssdCount++;
  } else {
    // 満杯なら最古を押し出し
    rmssdStart = (rmssdStart + 1) % RMSSD_MAX_BEATS;
  }

  // 1分より古いサンプルを先頭から除去
  while (rmssdCount > 0) {
    int idxOldest = rmssdStart;
    unsigned long oldestTime = rmssdBeatTimes[idxOldest];
    long age = (long)(beatTimeMs - oldestTime);
    if (age > (long)RMSSD_WINDOW_MS) {
      rmssdStart = (rmssdStart + 1) % RMSSD_MAX_BEATS;
      rmssdCount--;
    } else {
      break;
    }
  }
}

static float computeRmssdFromWindow() {
  if (rmssdCount < 2) return 0.0f;
  float sumSq = 0.0f;
  int nDiffs = 0;
  for (int i = 1; i < rmssdCount; ++i) {
    int idxPrev = (rmssdStart + i - 1) % RMSSD_MAX_BEATS;
    int idxCur  = (rmssdStart + i)     % RMSSD_MAX_BEATS;
    float d = (float)rmssdIbis[idxCur] - (float)rmssdIbis[idxPrev];
    sumSq += d * d;
    nDiffs++;
  }
  return sqrtf(sumSq / (float)nDiffs);
}
