import hypermedia.net.*;
import java.util.ArrayList;
import java.io.PrintWriter;  // CSV書き込み用

UDP udp;

// ウィンドウサイズや表示設定
int winWidth  = 1200;
int winHeight = 600;
int margin    = 50;

// バッファサイズ(横方向に表示する最大ドット数)
// ここでは「ウィンドウ幅 - マージン*2」あたりを目安に
int bufferSize = 1100;  

// 受信するデータ列数(初回受信で決定)
int numColumns = 0;

// データを保持する2次元配列
// waveData[col][i] = col番目の列の、左からi番目の描画用データ
float[][] waveData = new float[0][0];

// プログラム開始時刻(経過時間算出用)
int startMillis;

// CSV関連
String csvFilename;
PrintWriter output;

// ウィンドウ拡大・縮小用倍率
float resizeFactor = 1.1;

void settings() {
  size(winWidth, winHeight);
}

void setup() {
  // OS標準のウィンドウリサイズを許可
  surface.setResizable(true);

  // プログラム起動時のミリ秒を記録
  startMillis = millis();
  
  // ==================== CSVファイル名生成＆書き込み開始 ====================
  csvFilename = createTimestampFilename();   // "YYYYMMDD-HHMMSS.csv" 形式など
  output = createWriter(csvFilename);        // 書き込み先を準備
  
  // ==================== UDP初期化 ====================
  udp = new UDP(this, 8888);
  udp.listen(true);
  println("UDP Receiver started on port 8888");
  
  background(255);
}

void draw() {
  background(255);

  // ==================== ①現在時刻と経過時間を表示 ====================
  fill(0);
  textSize(16);

  // 現在時刻を文字列に変換 (例: "2025/03/04 12:34:56")
  String currentTime = year() + "/" 
                     + nf(month(), 2) + "/" 
                     + nf(day(), 2) + " " 
                     + nf(hour(), 2) + ":" 
                     + nf(minute(), 2) + ":" 
                     + nf(second(), 2);

  // プログラム開始からの経過秒数
  int elapsedMs = millis() - startMillis;
  float elapsedSec = elapsedMs / 1000.0;

  // 画面左上に2行で表示
  text(currentTime, 20, 30);
  text("Elapsed: " + nf(elapsedSec, 0, 2) + " sec", 20, 50);

  // ==================== グラフ枠描画 ====================
  stroke(0);
  noFill();
  rect(margin, margin, width - margin*2, height - margin*2);

  // まだデータが来ていなければ描画しない
  if (numColumns == 0) {
    fill(0);
    textSize(16);
    text("Waiting for data...", margin, height/2);
    return;
  }

  // 描画領域
  float graphWidth  = width  - margin * 2;
  float graphHeight = height - margin * 2;

  // 波形を表示する際のスケーリング範囲（必要に応じて調整）
  float minVal = -1000;
  float maxVal = 3000;

  // ========== 複数列を色を変えて描画 ==========
  for (int col = 0; col < numColumns; col++) {
    // 列ごとに色を変える
    stroke((col * 70) % 256, (col * 150) % 256, 200);
    strokeWeight(2);

    // waveData[col][i] を i=0 から i=bufferSize-2 まで順番に結ぶ
    for (int i = 0; i < bufferSize - 1; i++) {
      float x1 = map(i,   0, bufferSize - 1, margin, margin + graphWidth);
      float x2 = map(i+1, 0, bufferSize - 1, margin, margin + graphWidth);

      float y1 = map(waveData[col][i],   minVal, maxVal, 
                     margin + graphHeight, margin);
      float y2 = map(waveData[col][i+1], minVal, maxVal, 
                     margin + graphHeight, margin);

      line(x1, y1, x2, y2);
    }
  }

  // データ数を表示（右下）
  fill(0);
  textSize(14);
  text("Scroll Data (columns=" + numColumns + ")", margin, height - 10);
}

// ==================== UDP受信コールバック ====================
void receive(byte[] data, String ip, int port) {
  String message = new String(data).trim();
  println("Received from " + ip + ":" + port + " => " + message);

  // カンマ区切りで分割。先頭が空文字列(""、Arduinoからの先頭カンマ)ならスキップ
  String[] raw = message.split(",");
  int startIdx = (raw.length > 0 && raw[0].equals("")) ? 1 : 0;
  int dataCount = raw.length - startIdx;

  // 初回受信時に列数を決定し、配列を初期化 + ヘッダー出力
  if (numColumns == 0) {
    numColumns = dataCount;
    waveData = new float[numColumns][bufferSize];

    // Arduino側変数名に合わせてヘッダー行を出力
    // new_ecg_pt → RawECG  
    // diff       → DiffECG  
    // qrsFound   → QRSFlag  
    // timeDifference → RRInterval
    String[] headers = { "RawECG", "DiffECG", "QRSFlag", "RRInterval" };
    output.println(join(headers, ","));
  } 
  // 列数が想定と違えば無視
  else if (dataCount != numColumns) {
    println("列数が一致しません: 受信=" + dataCount + ", 想定=" + numColumns);
    return;
  }

  // 文字列配列 → float配列へ変換
  float[] currentValues = new float[numColumns];
  for (int i = 0; i < numColumns; i++) {
    try {
      currentValues[i] = Float.parseFloat(raw[i + startIdx]);
    } catch (NumberFormatException e) {
      println("数値変換エラー: " + raw[i + startIdx]);
      currentValues[i] = 0;
    }
  }

  // ========== スクロール(配列シフト)処理 ==========
  for (int col = 0; col < numColumns; col++) {
    for (int i = 0; i < bufferSize - 1; i++) {
      waveData[col][i] = waveData[col][i + 1];
    }
    // 右端(最新)に今回受信した値を入れる
    waveData[col][bufferSize - 1] = currentValues[col];
  }

  // ========== CSVファイルに1行出力 ==========
  // A列から順に書き込む
  String[] outStr = new String[numColumns];
  for (int i = 0; i < numColumns; i++) {
    outStr[i] = String.valueOf(currentValues[i]);
  }
  output.println(join(outStr, ","));
}

// ==================== CSVファイル名生成関数 ====================
String createTimestampFilename() {
  // YYYYMMDD-HHMMSS.csv のような文字列を生成
  String y  = str(year());
  String m  = nf(month(), 2);
  String d  = nf(day(), 2);
  String hh = nf(hour(), 2);
  String mm = nf(minute(), 2);
  String ss = nf(second(), 2);

  return y + m + d + "-" + hh + mm + ss + ".csv";
}

// ==================== 終了処理 ====================
void keyPressed() {
  // +キーでウィンドウ拡大
  if (key == '+') {
    int newW = int(width * resizeFactor);
    int newH = int(height * resizeFactor);
    surface.setSize(newW, newH);
  }
  // -キーでウィンドウ縮小
  else if (key == '-') {
    int newW = int(width / resizeFactor);
    int newH = int(height / resizeFactor);
    surface.setSize(newW, newH);
  }
  // ESCキーで終了
  else if (key == ESC) {
    // ファイルを確実に閉じて保存
    output.flush();
    output.close();
    println("CSV保存完了: " + csvFilename);
    exit();
  }
}
