#include <Wire.h>                  // I2C通信ライブラリ
#include <Adafruit_ADS1X15.h>      // ADS1015ライブラリ
#include <WiFi.h>                  // WiFiライブラリ（ESP32の場合）
#include <WiFiUdp.h>               // UDP通信ライブラリ

//=== プログラム2のパラメータ ============================================
#define M 5
#define N 30
#define winSize 250
#define HP_CONSTANT ((float)1 / (float)M)

//=== WiFi接続情報 ======================================================
const char* ssid = "～～～";
const char* password = "～～～";

//=== UDP送信先情報 =====================================================
const char* udpAddress = "192.168.68.104";  // 受信側PCのIPアドレスに変更
const int udpPort = 8888;                   // 使用するUDPポート番号

WiFiUDP udp;                // UDPオブジェクト
Adafruit_ADS1015 ads;       // ADS1015クラスのインスタンス化

//=== ボタン・ブザーなどのピン設定 =========================================
#define BUTTON_PIN 39         // M5Atom Liteのメインボタン用ピン（内部プルアップ）
#define BUZZER_PIN 19         // ブザー用ピン

//=== 平滑化微分用のパラメータ（プログラム1由来） ========================
const int WINDOW_SIZE = 16;                   // 微分計算用窓関数のサイズ
float voltageBuffer[WINDOW_SIZE];             // 電圧値のバッファ
// 窓関数の重み（例：左右対称で-8～-1, +1～+8）
float weights[WINDOW_SIZE] = {-8, -7, -6, -5, -4, -3, -2, -1, 1, 2, 3, 4, 5, 6, 7, 8};

//=== QRS検出・R-R間隔測定用変数 ========================================
unsigned long previousMillis = 0;  // 前回R波検出時刻（ms）
unsigned long timeDifference = 0;  // R-R間隔（ms）

//=== QRS検出用の関数 =========================================
boolean detect(float new_ecg_pt) {
  // static変数としてフィルタ用のバッファなどを保持
  static float ecg_buff[M + 1] = {0};
  static int   ecg_buff_WR_idx = 0;
  static int   ecg_buff_RD_idx = 0;
  static float hp_buff[N + 1] = {0};
  static int   hp_buff_WR_idx = 0;
  static int   hp_buff_RD_idx = 0;
  static float hp_sum = 0;
  static float lp_sum = 0;
  static float threshold = 0;
  static bool  triggered = false;
  static int   trig_time = 0;
  static float win_max = 0;
  static int   win_idx = 0;
  static int   number_iter = 0;

  // 生ECGデータをリングバッファに格納
  ecg_buff[ecg_buff_WR_idx++] = new_ecg_pt;
  ecg_buff_WR_idx %= (M + 1);

  // 簡易ハイパスフィルタ処理
  if (number_iter < M) {
    hp_sum += ecg_buff[ecg_buff_RD_idx];
    hp_buff[hp_buff_WR_idx] = 0.0;
  } else {
    hp_sum += ecg_buff[ecg_buff_RD_idx] - ecg_buff[(ecg_buff_RD_idx - M + (M + 1)) % (M + 1)];
    int centerIdx = (ecg_buff_RD_idx - (M + 1) / 2 + (M + 1)) % (M + 1);
    hp_buff[hp_buff_WR_idx] = ecg_buff[centerIdx] - HP_CONSTANT * hp_sum;
  }
  ecg_buff_RD_idx = (ecg_buff_RD_idx + 1) % (M + 1);
  hp_buff_WR_idx  = (hp_buff_WR_idx + 1) % (N + 1);

  // ローパス処理（2乗和の移動平均によるエネルギー計測）
  lp_sum += hp_buff[hp_buff_RD_idx] * hp_buff[hp_buff_RD_idx];
  if (number_iter >= N) {
    int oldIdx = (hp_buff_RD_idx - N + (N + 1)) % (N + 1);
    lp_sum -= hp_buff[oldIdx] * hp_buff[oldIdx];
  }
  hp_buff_RD_idx = (hp_buff_RD_idx + 1) % (N + 1);

  // 初期ウィンドウ内では閾値を設定
  if (number_iter < winSize) {
    if (lp_sum > threshold) {
      threshold = lp_sum;
    }
    number_iter++;
  }

  // 検出後の抑制処理
  if (triggered) {
    trig_time++;
    if (trig_time >= 100) {
      triggered = false;
      trig_time = 0;
    }
  }
  if (lp_sum > win_max) {
    win_max = lp_sum;
  }
  // QRS検出の条件：lp_sumが閾値を超えた場合
  if (lp_sum > threshold && !triggered) {
    triggered = true;
    return true;
  }

  // 一定サンプル毎に閾値更新
  if (++win_idx >= winSize) {
    float gamma = 0.175;
    float alpha = 0.01 + ((float)esp_random() / UINT32_MAX) * (0.1f - 0.01f);
    threshold = alpha * gamma * win_max + (1.0f - alpha) * threshold;
    win_idx = 0;
    win_max = -1.0e8;
  }
  return false;
}

//=== 平滑化微分関数（プログラム1由来） =============================
// ECG波形を微分し、滑らかな微分値を算出する
float smoothdiff(float newVoltage) {
  // バッファのシフト（最新値を先頭に）
  for (int i = WINDOW_SIZE - 1; i > 0; i--) {
    voltageBuffer[i] = voltageBuffer[i - 1];
  }
  voltageBuffer[0] = newVoltage;

  // 重み付き和による微分計算
  float result = 0;
  for (int i = 0; i < WINDOW_SIZE; i++) {
    result += voltageBuffer[i] * (-weights[i]);
  }
  return result / WINDOW_SIZE;
}

//=== セットアップ関数 =========================================
void setup(void) {
  pinMode(BUZZER_PIN, OUTPUT);
  pinMode(BUTTON_PIN, INPUT_PULLUP); // ボタンピンは内部プルアップ
  Serial.begin(115200);

  // ADS1015初期化（M5Atom用のI2Cピン設定）
  Wire.begin(21, 25);
  ads.begin();

  // WiFi接続
  WiFi.begin(ssid, password);
  Serial.print("Connecting to WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println();
  Serial.print("Connected, IP address: ");
  Serial.println(WiFi.localIP());

  // UDP初期化
  udp.begin(udpPort);

  // 微分用バッファの初期化
  for (int i = 0; i < WINDOW_SIZE; i++) {
    voltageBuffer[i] = 0.0;
  }
}

//=== ループ関数 =========================================
void loop(void) {
  // ボタン押下処理（必要に応じて）
  if (digitalRead(BUTTON_PIN) == LOW) {
    Serial.println("Pushed!");
    while (digitalRead(BUTTON_PIN) == LOW) {
      delay(10);
    }
    delay(50);
  }

  // ADS1015から差動入力取得
  int16_t results = ads.readADC_Differential_0_1();
  float multiplier = 3.0F; // ゲインに対応する係数
  float new_ecg_pt = -results * multiplier;  // 生ECG値（極性調整）

  // プログラム1の微分処理を適用
  float diff = smoothdiff(new_ecg_pt);

  // QRS検出アルゴリズムに投入
  bool qrsFound = detect(new_ecg_pt);
  if (qrsFound) {
    // R波検出時のタイミング計測（R-R間隔）
    unsigned long currentMillis = millis();
    timeDifference = currentMillis - previousMillis;
    previousMillis = currentMillis;

    // ブザーを一瞬ON
    digitalWrite(BUZZER_PIN, HIGH);
    delay(20);
    digitalWrite(BUZZER_PIN, LOW);

    Serial.print("QRS detected! R-R interval (ms) = ");
    Serial.println(timeDifference);
  }

  // UDP送信用メッセージ生成
  // 形式: ,生ECG値,微分値,QRS検出フラグ,R-R間隔
  String udpMessage = String(",") + String(new_ecg_pt, 2) + String(",") +
                      String(diff, 2) + String(",") +
                      String(qrsFound ? 1 : 0) + String(",") +
                      String(timeDifference);
                      
  Serial.println(udpMessage); // シリアル出力（デバッグ用）
  
  udp.beginPacket(udpAddress, udpPort);
  udp.print(udpMessage);
  udp.endPacket();

  delay(1);
}
