<?php
// showECG.php  (max modules = 8, download = today only)

/* ───── CONFIGURATION ───── */
$allModules = ['ECG01','ECG02','ECG03','ECG04','ECG05','ECG06','ECG07','ECG08'];
$maxMods    = 8;
$dataDir    = __DIR__ . '/Result/';
$rootDir    = __DIR__ . '/';

/* ───── CSV ダウンロード（当日分のみ） ───── */
if (isset($_GET['dl']) && in_array($_GET['dl'], $allModules, true)) {
    $csv  = $_GET['dl'] . '_' . date('Ymd') . '.csv';         // ← 今日固定
    $path = is_readable($dataDir.$csv) ? $dataDir.$csv
          : (is_readable($rootDir.$csv) ? $rootDir.$csv : null);
    if ($path) {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="'.$csv.'"');
        readfile($path);
    } else {
        http_response_code(404);
        echo 'file not found';
    }
    exit;
}

/* ───── GET / COOKIE / DEFAULTS ───── */
$date   = $_COOKIE['date']   ?? ($_GET['date']   ?? date('Ymd'));
$time   = $_COOKIE['time']   ?? ($_GET['time']   ?? date('His'));
$length = isset($_GET['length']) ? (int)$_GET['length'] : 600;
$min    = isset($_GET['min'])    ? (int)$_GET['min']    : ($_COOKIE['min'] ?? 500);
$max    = isset($_GET['max'])    ? (int)$_GET['max']    : ($_COOKIE['max'] ?? 1500);

/* ───── 〈★ 追加〉指標（IBI / HR / RMSSD）の取得 ───── */
$validMetrics = ['IBI','HR','RMSSD'];
/* ★ 追加：生波形 RAW を指標に追加（ドロップダウンに現れる） */
$validMetrics[] = 'RAW';
$metric = $_GET['metric'] ?? ($_COOKIE['metric'] ?? 'IBI');
if (!in_array($metric, $validMetrics, true)) $metric = 'IBI';
setcookie('metric', $metric, time()+86400*365, '/');    // 1年間保持

/* ───── validation & cookie ───── */
if (!preg_match('/^\d{8}$/',$date)) $date = date('Ymd');
if (!preg_match('/^\d{6}$/',$time)) $time = date('His');
if ($length<1) $length = 600;
if ($min<0)    $min    = 0;
if ($max <= $min) $max = $min + 100;
setcookie('date',$date);
setcookie('time',$time);
if(isset($_GET['min'])) setcookie('min',$min,time()+86400*365,'/');
if(isset($_GET['max'])) setcookie('max',$max,time()+86400*365,'/');

/* ───── module processing ───── */
$raw=[];
foreach($_GET as $k=>$v){
    if(preg_match('/^mod\d+$/',$k) && in_array($v,$allModules,true)) $raw[]=$v;
}
if($raw){
    $mods=array_slice(array_unique($raw),0,$maxMods);
    setcookie('modules',implode(',',$mods),time()+86400*365,'/');
}elseif(!empty($_COOKIE['modules'])){
    $mods=array_values(array_intersect($allModules,explode(',',$_COOKIE['modules'])));
}else{
    $mods=array_slice($allModules,0,$maxMods);
}
if(!$mods) $mods=array_slice($allModules,0,$maxMods);

/* ───── 〈★ 追加〉列番号マップ（ts1, ts2, IBI, HR, RMSSD） ───── */
$colMap = ['IBI'=>2, 'HR'=>3, 'RMSSD'=>4];

/* ★ 追加：RAW専用のAJAX応答（既存AJAXの手前で早期return） */
if (isset($_GET['ajax']) && $_GET['ajax'] === '1' && ($metric === 'RAW')) {
    header('Content-Type: application/json');

    // now=1 なら現在時刻を使用／time=HHMMSS を手動指定可
    if (isset($_GET['now']) && $_GET['now'] === '1') {
        $time = date('His');
    } elseif (isset($_GET['time']) && preg_match('/^\d{6}$/', $_GET['time'])) {
        $time = $_GET['time'];
    }
    setcookie('time', $time);

    // 表示窓（ms）
    $dtEnd = DateTime::createFromFormat('YmdHis', $date . $time);
    if (!$dtEnd) { echo json_encode(['error'=>'time parse error']); exit; }
    $toMs   = $dtEnd->getTimestamp() * 1000;
    $fromMs = $toMs - ($length * 1000);

    // 各モジュールのCSVから "RAW,<ts_ms>,<mv>" を抽出
    $series = [];
    foreach ($mods as $mod) {
        $csv  = "{$mod}_{$date}.csv";
        $file = is_readable($dataDir . $csv) ? $dataDir . $csv
               : (is_readable($rootDir . $csv) ? $rootDir . $csv : null);

        $pts = [];
        if ($file && ($fh = fopen($file, 'r'))) {
            while (($ln = fgets($fh)) !== false) {
                if (strncmp($ln, 'RAW,', 4) !== 0) continue;
                $ln  = trim($ln);
                $col = explode(',', $ln);
                if (count($col) < 3) continue;
                $ts = (int)preg_replace('/[^0-9]/','', $col[1]);
                if ($ts < $fromMs || $ts > $toMs) continue;
                $mv = (float)$col[2];
                $pts[] = ['ts_ms' => $ts, 'y' => $mv];
            }
            fclose($fh);
        }
        usort($pts, fn($a,$b)=>$a['ts_ms'] <=> $b['ts_ms']);
        $series[$mod] = $pts;
    }

    // RAWの既定Y軸（mV）。明示指定があればそちらを優先。
    if (!isset($_GET['min']) && !isset($_GET['max'])) {
        // IBI等の既定(500-1500)が残っている場合のみ置換
        if ($min >= 500 || $max >= 500) { $min = -1.5; $max = 1.5; }
    }

    echo json_encode([
        'date'   => $date,
        'time'   => $time,
        'length' => $length,
        'min'    => $min,
        'max'    => $max,
        'mods'   => $mods,
        'series' => $series,
        'metric' => $metric
    ]);
    exit;
}

/* ───── AJAX endpoint ───── */
if (isset($_GET['ajax']) && $_GET['ajax'] === '1') {
    header('Content-Type: application/json');

    /* now=1 なら現在時刻を使用／time=HHMMSS を手動指定可 */
    if (isset($_GET['now']) && $_GET['now'] === '1') {
        $time = date('His');
    } elseif (isset($_GET['time']) && preg_match('/^\d{6}$/', $_GET['time'])) {
        $time = $_GET['time'];
    }
    setcookie('time', $time);

    $targetSec = intval(substr($time, 0, 2)) * 3600
               + intval(substr($time, 2, 2)) * 60
               + intval(substr($time, 4, 2));
    $fromSec = $targetSec - $length;

    $series = [];
    foreach ($mods as $mod) {
        $csv  = "{$mod}_{$date}.csv";
        $file = is_readable($dataDir . $csv) ? $dataDir . $csv
               : (is_readable($rootDir . $csv) ? $rootDir . $csv : null);

        $pts = [];
        if ($file && ($fh = fopen($file, 'r'))) {
            while (($L = fgetcsv($fh)) !== false) {
                /* ts1, ts2, IBI, HR, RMSSD → 5 列必要 */
                if (count($L) < 5) continue;

                /* ───── ① ts1 を数値だけ残し、先頭 8 桁(日付)を抜き出す ───── */
                $ts1Clean = preg_replace('/[^0-9]/', '', $L[0]);      // 例: 20250805141909204
                $datePart = substr($ts1Clean, 0, 8);                  // 例: 20250805

                /* ───── ② ts2 も数値化して HHMMSS にする ───── */
                $ts2Clean = preg_replace('/[^0-9]/', '', $L[1]);      // 例: 141909
                if (strlen($datePart) !== 8 || strlen($ts2Clean) < 6) continue;

                /* CSV の日付がフォーム指定と異なる行はスキップ（任意だが安全） */
                if ($datePart !== $date) continue;

                $h = substr($ts2Clean, 0, 2);
                $i = substr($ts2Clean, 2, 2);
                $s = substr($ts2Clean, 4, 2);
                $sec = $h * 3600 + $i * 60 + $s;
                if ($sec < $fromSec || $sec > $targetSec) continue;

                /* ───── ③ DateTime 生成 → epoch(ms) ───── */
                $dt = DateTime::createFromFormat('YmdHis', $datePart . $h . $i . $s);
                if (!$dt) continue;
                $epoch = $dt->getTimestamp() * 1000;                 // ms 精度不要

                /* ───── ④ 選択指標 (IBI/HR/RMSSD) の列を取得 ───── */
                $val = isset($L[$colMap[$metric]]) ? (float)$L[$colMap[$metric]] : null;
                if ($val === null || $val === 0) continue;           // 0 は欠損扱い
                $pts[] = ['ts_ms' => $epoch, 'y' => $val];
            }
            fclose($fh);
        }
        $series[$mod] = $pts;   // モジュールごとの系列を格納
    }

    /* ───── JSON でクライアントへ返却 ───── */
    echo json_encode([
        'date'   => $date,
        'time'   => $time,
        'length' => $length,
        'min'    => $min,
        'max'    => $max,
        'mods'   => $mods,
        'series' => $series,
        'metric' => $metric       /* 選択指標をクライアントへ通知 */
    ]);
    exit;
}

?>
<!DOCTYPE html>
<html>
<head>
<meta charset="utf-8">
<title>ECG Plot</title>
<script src="https://www.gstatic.com/charts/loader.js"></script>
<style>
 body{font-family:sans-serif;margin:2em;}
 form{margin-bottom:1em;}
 .row{display:flex;align-items:center;margin-bottom:0.5em;flex-wrap:wrap;}
 .row input{width:6em;margin-right:1em;}
 button{margin:0.2em;}
 .textarea-container{display:flex;gap:1em;margin-top:1em;flex-wrap:wrap;}
 .textarea-container .block{flex:1 1 45%;min-width:260px;}
 .block textarea{width:100%;height:150px;}
 .block .dl-link{display:block;margin-bottom:0.3em;}
</style>
</head>
<body>
<h2 id="pageTitle">ECG Plot</h2>

<form id="cfgForm">
  <div class="row">
    Date:<input name="date" value="<?php echo $date ?>">
    EndTime:<input name="time" value="<?php echo $time ?>" id="endTimeField">
    Length:<input name="length" value="<?php echo $length ?>">
  </div>
  <div class="row">
    Min:<input name="min" value="<?php echo $min ?>">
    Max:<input name="max" value="<?php echo $max ?>">
    <!-- 〈★ 追加〉指標ドロップダウン -->
    <label style="margin-left:1em;">指標：
      <select name="metric" id="metricSel">
        <?php foreach($validMetrics as $m): ?>
          <option value="<?php echo $m ?>" <?php if($m===$metric) echo 'selected'; ?>>
            <?php echo $m ?>
          </option>
        <?php endforeach; ?>
      </select>
    </label>
  </div>
  <div class="row" id="modsContainer"></div>
  <button type="button" id="addModBtn">＋ Module</button>
  <button type="button" id="delModBtn">− Module</button><br>
  <button type="button" id="startBtn">Start Auto-Refresh</button>
  <button type="button" id="stopBtn">Stop Auto-Refresh</button>
  <button type="button" id="applyBtn">Apply</button>
</form>

<div id="chart_div" style="width:100%;height:450px;"></div>
<div class="textarea-container" id="taContainer"></div>

<script>
google.charts.load('current',{packages:['corechart']});
let chart,dataTable,options;
let statusLabel='(refresh)',intervalId=null;

/* === dynamic module inputs === */
const modsMax = 8;
const phpMods = <?php echo json_encode($mods); ?>;
let modCount  = phpMods.length;

function mkInput(idx,val=''){
  const inp=document.createElement('input');
  inp.name='mod'+idx; inp.value=val; inp.placeholder='ECGxx';
  inp.style.width='6em'; inp.style.marginRight='1em';
  return inp;
}
function rebuildMods(){
  const mc=document.getElementById('modsContainer');
  mc.textContent='';
  for(let i=1;i<=modCount;i++){
    mc.appendChild(document.createTextNode(i+': '));
    mc.appendChild(mkInput(i,phpMods[i-1]??''));
  }
  rebuildBlocks();
}
document.getElementById('addModBtn').onclick=()=>{if(modCount<modsMax){modCount++;rebuildMods();}};
document.getElementById('delModBtn').onclick=()=>{if(modCount>1){modCount--;rebuildMods();}};
rebuildMods();

/* === textarea & download blocks === */
function rebuildBlocks(){
  const tc=document.getElementById('taContainer');
  tc.textContent='';
  for(let i=1;i<=modsMax;i++){
    const block=document.createElement('div'); block.className='block';
    const link=document.createElement('a'); link.className='dl-link'; link.id='dl'+i;
    link.textContent='Download CSV (today)'; link.target='_blank';
    const ta=document.createElement('textarea'); ta.name='text'+i;
    if(i>modCount){ block.style.display='none'; }
    block.appendChild(link); block.appendChild(ta); tc.appendChild(block);
  }
}

/* === fetch / chart handling === */
async function fetchData(auto){
  const p=new URLSearchParams(new FormData(document.getElementById('cfgForm')));
  p.set('ajax','1');
  if(auto){p.set('now','1');p.delete('time');}
  const r=await fetch('showECG.php?'+p); return r.json();
}

function initChart(j){
  dataTable=new google.visualization.DataTable();
  dataTable.addColumn('datetime','Time');
  j.mods.forEach(m=>dataTable.addColumn('number',m));
  new google.visualization.DateFormat({pattern:'HH:mm:ss'}).format(dataTable,0);
  options={
    hAxis:{title:'Time',format:'HH:mm:ss'},
    vAxis:{title:getYAxisTitle(j.metric)},
    legend:{position:'right'}, pointSize:3,
    interpolateNulls:true,
    explorer:{actions:['dragToZoom','rightClickToReset'],axis:'horizontal'}
  };
  chart=new google.visualization.LineChart(document.getElementById('chart_div'));
  updateChart(j);
}

function getYAxisTitle(metric){
  switch(metric){
    case 'HR'   : return 'Heart Rate (bpm)';
    case 'RMSSD': return 'RMSSD (ms)';
    default     : return 'IBI (ms)';
  }
}

function updateChart(j){
  document.getElementById('endTimeField').value=j.time;
  /* 〈★ 追加〉ドロップダウンの選択を同期 */
  document.getElementById('metricSel').value=j.metric;

  const rows=dataTable.getNumberOfRows(); if(rows) dataTable.removeRows(0,rows);
  const map={};
  j.mods.forEach(m=>{
    (j.series[m]||[]).forEach(pt=>{
      if(!map[pt.ts_ms]) map[pt.ts_ms]={};
      map[pt.ts_ms][m]=pt.y;
    });
  });
  Object.keys(map).map(Number).sort((a,b)=>a-b).forEach(ts=>{
    const row=[new Date(ts)]; j.mods.forEach(m=>row.push(map[ts][m]??null)); dataTable.addRow(row);
  });

  options.title=`${j.mods.join(', ')} – ${j.date} ${j.time} ±${j.length}s ${statusLabel}`;
  options.vAxis.title=getYAxisTitle(j.metric);
  options.vAxis.viewWindow={min:j.min,max:j.max};
  document.getElementById('pageTitle').textContent=`ECG Plot ${statusLabel}`;
  chart.draw(dataTable,options);

  /* textarea & link update */
  const blocks=document.querySelectorAll('#taContainer .block');
  j.mods.forEach((mod,idx)=>{
    const link=blocks[idx].querySelector('.dl-link');
    link.href=`showECG.php?dl=${mod}`;              // ← 当日固定
    const ta=blocks[idx].querySelector('textarea');
    const lines=Object.keys(map).map(Number).sort((a,b)=>a-b).flatMap(ts=>{
      const v=map[ts][mod]; if(v==null) return [];
      const d=new Date(ts);
      return `${d.toLocaleTimeString('en-GB',{hour12:false})},${v}`;
    });
    ta.value=lines.length?lines.join('\n'):'データがありません';
  });
}

function startAuto(){ if(intervalId) return; statusLabel='(refresh)'; fetchData(true).then(h); intervalId=setInterval(()=>fetchData(true).then(h),5000);}
function stopAuto(){ if(!intervalId) return; clearInterval(intervalId); intervalId=null; statusLabel='(stop)'; fetchData(false).then(h);}
function h(j){ chart?updateChart(j):initChart(j); }

document.getElementById('startBtn').onclick=startAuto;
document.getElementById('stopBtn').onclick=stopAuto;
document.getElementById('applyBtn').onclick=()=>{stopAuto(); fetchData(false).then(h);};

google.charts.setOnLoadCallback(async()=>{const j=await fetchData(true); initChart(j); startAuto();});

/* ★ 追加：RAW時はY軸タイトルを mV に切替（既存関数を薄くラップ） */
(function(prev){
  window.getYAxisTitle = function(metric){
    if (metric === 'RAW') return 'Amplitude (mV)';
    return prev(metric);
  };
})(getYAxisTitle);
</script>
</body>
</html>
