// M5StickC 初期型（MPU6886） Pitch / Roll 取得デモ
// 50Hz更新，相補フィルタ(α=0.98)
// 画面表示 + シリアルCSV出力（Pitch,Roll）

#include <M5StickC.h>

const float alpha = 0.98f;      // 相補フィルタ係数（0.95〜0.99で調整）
const uint32_t dt_ms = 20;      // サンプリング周期 20ms → 50Hz
const float dt = dt_ms / 1000.0f;

float pitch = 0.0f;             // 融合後の角度[deg]
float roll  = 0.0f;

void setup() {
  M5.begin();
  M5.IMU.Init();                // MPU6886 初期化
  M5.Lcd.setRotation(3);        // 横向き
  M5.Lcd.fillScreen(BLACK);
  M5.Lcd.setTextSize(2);
  M5.Lcd.setCursor(0, 0);
  M5.Lcd.println("M5StickC IMU");
  M5.Lcd.println("Pitch/Roll");

  Serial.begin(115200);
  while (!Serial) { delay(10); }

  // 初期姿勢を加速度ベースで推定して，初期値に採用
  float ax, ay, az;
  M5.IMU.getAccelData(&ax, &ay, &az);        // [g]
  float pitchAcc = atan2f(-ax, sqrtf(ay*ay + az*az)) * 180.0f / PI;
  float rollAcc  = atan2f( ay, az ) * 180.0f / PI;
  pitch = pitchAcc;
  roll  = rollAcc;

  delay(500);
}

void loop() {
  // センサ取得
  float ax, ay, az;           // 加速度[g]
  float gx, gy, gz;           // 角速度[deg/s]
  M5.IMU.getAccelData(&ax, &ay, &az);
  M5.IMU.getGyroData(&gx, &gy, &gz);

  // 加速度からの姿勢角（重力のみ仮定）
  float pitchAcc = atan2f(-ax, sqrtf(ay*ay + az*az)) * 180.0f / PI;
  float rollAcc  = atan2f( ay, az ) * 180.0f / PI;

  // ジャイロ積分（軸対応：MPU6886はおおむね gx→Roll, gy→Pitch）
  float pitchGyro = pitch + gy * dt;
  float rollGyro  = roll  + gx * dt;

  // 相補フィルタ融合
  pitch = alpha * pitchGyro + (1.0f - alpha) * pitchAcc;
  roll  = alpha * rollGyro  + (1.0f - alpha) * rollAcc;

  // 画面表示
  M5.Lcd.fillRect(0, 40, 160, 40, BLACK);
  M5.Lcd.setCursor(0, 40);
  M5.Lcd.printf("Pitch:%6.2f deg\n", pitch);
  M5.Lcd.printf("Roll :%6.2f deg\n", roll);

  // シリアルCSV（Pitch,Roll）
  Serial.print(pitch, 3);
  Serial.print(",");
  Serial.println(roll, 3);

  delay(dt_ms);
}
