// M5StickC 初期型（MPU6886） Pitch / Roll 取得＋Pitch>10°でG26出力
// 相補フィルタ(α=0.98) / 50Hz / ヒステリシス: ON 10°, OFF 8°

#include <M5StickC.h>

const int MOTOR_PIN = 26;          // G26 にモータードライバの入力を接続（直結は不可）
const float alpha = 0.98f;         // 相補フィルタ係数
const uint32_t dt_ms = 20;         // 50Hz
const float dt = dt_ms / 1000.0f;

const float PITCH_ON_DEG  = 10.0f; // これを超えたらON
const float PITCH_OFF_DEG = 8.0f;  // これを下回ったらOFF（ヒステリシス）

float pitch = 0.0f;
float roll  = 0.0f;
bool motorOn = false;

void setup() {
  M5.begin();
  M5.IMU.Init();
  M5.Lcd.setRotation(3);
  M5.Lcd.fillScreen(BLACK);
  M5.Lcd.setTextSize(2);
  M5.Lcd.setCursor(0, 0);
  M5.Lcd.println("M5StickC IMU");
  M5.Lcd.println("Pitch->G26");

  Serial.begin(115200);
  while (!Serial) delay(10);

  // モーターピン初期化
  pinMode(MOTOR_PIN, OUTPUT);
  digitalWrite(MOTOR_PIN, LOW);   // 初期はOFF

  // 初期姿勢（加速度ベース）
  float ax, ay, az;
  M5.IMU.getAccelData(&ax, &ay, &az);
  float pitchAcc = atan2f(-ax, sqrtf(ay*ay + az*az)) * 180.0f / PI;
  float rollAcc  = atan2f( ay, az ) * 180.0f / PI;
  pitch = pitchAcc;
  roll  = rollAcc;

  delay(300);
}

void loop() {
  // センサ取得
  float ax, ay, az;  // [g]
  float gx, gy, gz;  // [deg/s]
  M5.IMU.getAccelData(&ax, &ay, &az);
  M5.IMU.getGyroData(&gx, &gy, &gz);

  // 姿勢角（加速度）
  float pitchAcc = atan2f(-ax, sqrtf(ay*ay + az*az)) * 180.0f / PI;
  float rollAcc  = atan2f( ay, az ) * 180.0f / PI;

  // ジャイロ積分
  float pitchGyro = pitch + gy * dt;  // gy ≈ pitch軸
  float rollGyro  = roll  + gx * dt;  // gx ≈ roll軸

  // 相補フィルタ融合
  pitch = alpha * pitchGyro + (1.0f - alpha) * pitchAcc;
  roll  = alpha * rollGyro  + (1.0f - alpha) * rollAcc;

  // —— Pitchしきい値でG26制御（ヒステリシスあり）——
  if (!motorOn && pitch > PITCH_ON_DEG) {
    motorOn = true;
    digitalWrite(MOTOR_PIN, HIGH);
  } else if (motorOn && pitch < PITCH_OFF_DEG) {
    motorOn = false;
    digitalWrite(MOTOR_PIN, LOW);
  }

  // 表示＆ログ
  M5.Lcd.fillRect(0, 40, 160, 50, BLACK);
  M5.Lcd.setCursor(0, 40);
  M5.Lcd.printf("Pitch:%6.2f deg\n", pitch);
  M5.Lcd.printf("Roll :%6.2f deg\n", roll);
  M5.Lcd.printf("MOTOR:%s    \n", motorOn ? "ON " : "OFF");

  Serial.print(pitch, 3);
  Serial.print(",");
  Serial.print(roll, 3);
  Serial.print(",");
  Serial.println(motorOn ? 1 : 0);

  delay(dt_ms);
}
