#include <Arduino.h>
#include <Wire.h>
constexpr uint8_t LSM6_ADDR_GND = 0x6A;
constexpr uint8_t LSM6_ADDR_VCC = 0x6B;
uint8_t lsm6_addr = 0;

constexpr uint8_t REG_WHO_AM_I  = 0x0F;
constexpr uint8_t REG_CTRL1    = 0x10;
constexpr uint8_t REG_CTRL2    = 0x11;
constexpr uint8_t REG_CTRL6    = 0x15;
constexpr uint8_t REG_CTRL8    = 0x17;
constexpr uint8_t REG_OUTX_L_G = 0x22;

// ---------------- I2C helpers ----------------
void writeReg(uint8_t reg, uint8_t val) {
  Wire.beginTransmission(lsm6_addr);
  Wire.write(reg);
  Wire.write(val);
  Wire.endTransmission();
}

bool readRegsOnce(uint8_t startReg, uint8_t* buf, uint8_t len) {
  Wire.beginTransmission(lsm6_addr);
  Wire.write(startReg);
  uint8_t e = Wire.endTransmission(true);  // STOP
  if (e != 0) return false;

  //delayMicroseconds(50);
  uint8_t got = Wire.requestFrom((int)lsm6_addr, (int)len);
  if (got != len) return false;

  for (uint8_t i = 0; i < len; i++) buf[i] = Wire.read();
  return true;
}

bool readRegs(uint8_t startReg, uint8_t* buf, uint8_t len) {
  // 2～3回リトライ（軽量）
  for (int attempt = 0; attempt < 3; attempt++) {
    if (readRegsOnce(startReg, buf, len)) return true;

    // 失敗時：少し待って再試行
    delay(2);

    // バス/ドライバの再初期化で復帰することがある（ESP32で有効なことが多い）
    Wire.end();
    delay(2);
    //Wire.begin(D4, D5);       // XIAO ESP32S3: SDA=D4, SCL=D5
    Wire.begin(26, 0);       // M5StickC: SDA=26, SCL=0
    Wire.setClock(50000);
    delay(2);
  }
  return false;
}

// ---------------- LSM6 init ----------------
bool lsm6_init() {
  
  uint8_t who = 0;
  if (!readRegs(REG_WHO_AM_I, &who, 1)) return false;
  Serial.print("WHO_AM_I = 0x"); Serial.println(who, HEX);

  if (who != 0x70) {
    return false;
  }

  // 初期設定（秋月サンプルそのまま）
  writeReg(REG_CTRL1, 0b00001001);
  writeReg(REG_CTRL2, 0b00001001);
  writeReg(REG_CTRL6, 0b00000100);
  writeReg(REG_CTRL8, 0b10000010);
  delay(10);
  
  return true;
}

// ---------------- Read sensor ----------------
bool readSensor(float& gx, float& gy, float& gz, float& ax, float& ay, float& az) {
  uint8_t buf[12];
  int16_t raw[6];

  if (!readRegs(REG_OUTX_L_G, buf, 12)) {
    return false;  // この周期は捨てる
  }

  for (int i = 0; i < 6; i++) {
    raw[i] = (int16_t)(buf[i * 2 + 1] << 8 | buf[i * 2]);
  }

  gx = raw[0] * 0.07f;
  gy = raw[1] * 0.07f;
  gz = raw[2] * 0.07f;
  ax = raw[3] * 0.000244f;
  ay = raw[4] * 0.000244f;
  az = raw[5] * 0.000244f;
  return true;
}


//////////////////////////////////////
void setup() {
  Serial.begin(115200);
  delay(500);
  Serial.println("BOOT");

  
  //Wire.begin(D4, D5);       // XIAO ESP32S3: SDA=D4, SCL=D5
  Wire.begin(D2, D1);       // XIAO ESP32C3: SDA=D2, SCL=D1
  //Wire.begin(26, 0);       // M5StickC: SDA=26, SCL=0
  Wire.setClock(50000);

  Serial.println("\nStart!");

  // アドレス探索（0x6A → 0x6B）
  Wire.beginTransmission(LSM6_ADDR_GND);
  if (Wire.endTransmission() == 0) {
    Serial.print("LSM6DSV16X found at 0x");
    Serial.println(LSM6_ADDR_GND, HEX);
    lsm6_addr = LSM6_ADDR_GND;
  }

  Wire.beginTransmission(LSM6_ADDR_VCC);
  if (Wire.endTransmission() == 0) {
    Serial.print("LSM6DSV16X found at 0x");
    Serial.println(LSM6_ADDR_VCC, HEX);
    lsm6_addr = LSM6_ADDR_VCC;
  }
  if (lsm6_addr == 0) {
    Serial.println("LSM6DSV16X not found");
    while (1);
  }

  
  if (!lsm6_init()) {
    Serial.println("LSM6DSV16X init error!");
    while (1);
  }

  Serial.println("LSM6DSV16X initialized.");  
  
}

void loop() {
  float ax, ay, az, gx, gy, gz;

  if (readSensor(gx, gy, gz, ax, ay, az)) {
    Serial.print(ax); Serial.print(", ");
    Serial.print(ay); Serial.print(", ");
    Serial.print(az); Serial.print(", ");
    Serial.print(gx); Serial.print(", ");
    Serial.print(gy); Serial.print(", ");
    Serial.println(gz);
  } else {
    Serial.println("readSensor failed (skip)");
  }

  delay(20);
}
