#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiClient.h>

#include <Wire.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>

// ====== Wi-Fi設定 ======
const char* ssid     = "HELL";
const char* password = "kodamamasahisa";

// ====== 送信先URL ======
const char* endpoint = "http://kodamalab.sakura.ne.jp/WIFI_BME_8266/nagano/receiveBME.php";

// ====== I2Cピン（必要なら変更） ======
static const uint8_t SDA_PIN = 4; // GPIO4
static const uint8_t SCL_PIN = 5; // GPIO5

Adafruit_BME280 bme;
WiFiClient client;

unsigned long lastSendMs = 0;
const unsigned long intervalMs = 1000; // 1秒

// ====== A0(ADC) 設定 ======
// ★あなたのボードに合わせて調整してください★
//
// NodeMCU / Wemos D1 mini など：A0が 0～3.3V 相当にスケールされていることが多い
// ESP-12E 直結ADC(分圧なし)など：0～1.0V の場合がある
const float ADC_MAX_VOLT = 3.0f;   //A0入力は1/3されてることを想定。
const int   ADC_MAX_RAW  = 1023;

float readA0Voltage() {
  int raw = analogRead(A0); // 0..1023
  return (raw * ADC_MAX_VOLT) / ADC_MAX_RAW;
}

void connectWiFi() {
  Serial.println();
  Serial.println("WiFiに接続中…");

  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, password);

  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println();
  Serial.println("WiFi接続成功！");
  Serial.print("IPアドレス: ");
  Serial.println(WiFi.localIP());
}

bool initBME280() {
  Wire.begin(SDA_PIN, SCL_PIN);
  Wire.setClock(100000);

  bool ok = bme.begin(0x76);
  if (!ok) ok = bme.begin(0x77);
  if (!ok) return false;

  bme.setSampling(
    Adafruit_BME280::MODE_NORMAL,
    Adafruit_BME280::SAMPLING_X2,   // Temp
    Adafruit_BME280::SAMPLING_X16,  // Pressure
    Adafruit_BME280::SAMPLING_X1,   // Humidity
    Adafruit_BME280::FILTER_X16,
    Adafruit_BME280::STANDBY_MS_500
  );

  return true;
}

void setup() {
  Serial.begin(115200);
  delay(1000);

  connectWiFi();

  if (!initBME280()) {
    Serial.println("BME280 not found. 配線/アドレス(0x76/0x77)を確認してください。");
    while (true) delay(1000);
  }
  Serial.println("BME280 OK");

  // A0の初期値確認（任意）
  Serial.print("A0 voltage(V): ");
  Serial.println(readA0Voltage(), 3);
}

void loop() {
  // Wi-Fi切断時は復帰
  if (WiFi.status() != WL_CONNECTED) {
    Serial.println("WiFi切断。再接続します…");
    connectWiFi();
  }

  // 1秒ごとに送信
  unsigned long now = millis();
  if (now - lastSendMs < intervalMs) return;
  lastSendMs = now;

  float tempC = bme.readTemperature(); // ℃
  if (isnan(tempC)) {
    Serial.println("温度読み取り失敗");
    return;
  }

  float batV = readA0Voltage(); // A0電圧(V)

  HTTPClient http;

  // 例: ...receiveBME.php?temp=23.45&bat=0.92
  String url = String(endpoint)
             + "?temp=" + String(tempC, 2)
             + "&bat="  + String(batV, 2);

  Serial.print("GET: ");
  Serial.println(url);

  http.begin(client, url);
  http.setTimeout(5000);

  int code = http.GET();
  if (code > 0) {
    String payload = http.getString();
    Serial.print("HTTP ");
    Serial.print(code);
    Serial.print(" / ");
    Serial.println(payload);
  } else {
    Serial.print("HTTP error: ");
    Serial.println(http.errorToString(code));
  }
  http.end();
}
