#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiClient.h>

#include <Wire.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>

// ====== 各種設定 ======
const char* ssid     = "HELL";
const char* password = "kodamamasahisa";
const char* modName  = "IoT8266_01";    // モジュール名
const char* staticIP = "192.168.68.250"; // ←ここを変更するだけ

// ====== ネットワーク固定値 ======
const char* gatewayIP = "192.168.68.1";
const char* subnetMask = "255.255.255.0";
const char* dnsIP = "192.168.68.1";

// ====== 送信先URL ======
const char* endpoint =
  "http://kodamalab.sakura.ne.jp/WIFI_BME_8266/nagano/receiveBME.php";

// ====== I2Cピン ======
static const uint8_t SDA_PIN = 4;
static const uint8_t SCL_PIN = 5;

// ====== LEDピン ======
static const uint8_t LED_PIN = 14;

// ====== DeepSleep間隔 ======
static const uint64_t SLEEP_SEC = 60;

// ====== A0(ADC) 設定 ======
const float ADC_MAX_VOLT = 3.202f;
const int   ADC_MAX_RAW  = 1023;

Adafruit_BME280 bme;
WiFiClient client;


// ===== URLエンコード =====
String urlEncode(const String& s) {
  String out;
  const char *hex = "0123456789ABCDEF";
  for (size_t i = 0; i < s.length(); i++) {
    uint8_t c = (uint8_t)s[i];
    if (isalnum(c) || c=='_' || c=='-' || c=='.') {
      out += (char)c;
    } else {
      out += '%';
      out += hex[(c >> 4) & 0xF];
      out += hex[c & 0xF];
    }
  }
  return out;
}


// ===== A0電圧取得 =====
float readA0Voltage() {
  int raw = analogRead(A0);
  return (raw * ADC_MAX_VOLT) / ADC_MAX_RAW;
}


// ===== 固定IP接続 =====
bool connectWiFiStatic(uint32_t timeoutMs = 15000) {

  WiFi.mode(WIFI_STA);

  IPAddress local_IP;
  IPAddress gateway;
  IPAddress subnet;
  IPAddress dns;

  local_IP.fromString(staticIP);
  gateway.fromString(gatewayIP);
  subnet.fromString(subnetMask);
  dns.fromString(dnsIP);

  if (!WiFi.config(local_IP, gateway, subnet, dns)) {
    Serial.println("Static IP設定失敗");
  }

  WiFi.begin(ssid, password);

  Serial.print("WiFiに接続中…");
  uint32_t start = millis();

  while (WiFi.status() != WL_CONNECTED) {
    delay(250);
    Serial.print(".");
    if (millis() - start > timeoutMs) {
      Serial.println("\nWiFi接続タイムアウト");
      return false;
    }
  }

  Serial.println("\nWiFi接続成功！");
  Serial.print("IPアドレス: ");
  Serial.println(WiFi.localIP());
  return true;
}


// ===== BME初期化 =====
bool initBME280() {
  Wire.begin(SDA_PIN, SCL_PIN);
  Wire.setClock(100000);

  bool ok = bme.begin(0x76);
  if (!ok) ok = bme.begin(0x77);
  if (!ok) return false;

  bme.setSampling(
    Adafruit_BME280::MODE_FORCED,
    Adafruit_BME280::SAMPLING_X2,
    Adafruit_BME280::SAMPLING_X16,
    Adafruit_BME280::SAMPLING_X1,
    Adafruit_BME280::FILTER_X16,
    Adafruit_BME280::STANDBY_MS_500
  );
  return true;
}


// ===== DeepSleep =====
void goSleep() {
  digitalWrite(LED_PIN, LOW);
  Serial.println("DeepSleepへ移行します…");
  Serial.flush();

  WiFi.disconnect(true);
  WiFi.mode(WIFI_OFF);

  ESP.deepSleep(SLEEP_SEC * 1000000ULL, WAKE_RF_DEFAULT);
  delay(100);
}


// ===== setup =====
void setup() {
  Serial.begin(115200);
  delay(50);

  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, HIGH);

  Serial.println("\n起床しました");

  if (!initBME280()) {
    Serial.println("BME280 not found.");
    goSleep();
  }

  if (!connectWiFiStatic()) {
    goSleep();
  }

  bme.takeForcedMeasurement();
  float tempC = bme.readTemperature();
  if (isnan(tempC)) {
    Serial.println("温度読み取り失敗");
    goSleep();
  }

  delay(50);
  float batV = readA0Voltage();

  Serial.print("tempC: ");
  Serial.print(tempC, 2);
  Serial.print(" / batV: ");
  Serial.println(batV, 2);

  HTTPClient http;

  String url = String(endpoint)
             + "?temp=" + String(tempC, 2)
             + "&bat="  + String(batV, 2)
             + "&mod="  + urlEncode(String(modName));

  Serial.print("GET: ");
  Serial.println(url);

  http.begin(client, url);
  http.setTimeout(8000);

  int code = http.GET();
  if (code > 0) {
    Serial.print("HTTP ");
    Serial.println(code);
  } else {
    Serial.print("HTTP error: ");
    Serial.println(http.errorToString(code));
  }

  http.end();
  goSleep();
}

void loop() {}
